/* 
   CC0 2011, Martin Haye

   To the extent possible under law, Martin Haye has waived all copyright 
   and related or neighboring rights to p2e: Pseudo-II Emulator. 
   This work is published from: United States.
*/

// Support for Apple Disk II images

// Constructor
function DiskDrive() 
{
  var quiet = false;
  
  this.setQuiet = function(flg) {
    quiet = flg;
  }
  
  function log(msg) {
    if (!quiet)
      console.debug(msg);
  }
  
  /* ======================================================================= */
  
  /** Binary search an array. Returns proper insertion pos of val */
  function binarySearch(array, val)
  {
    var i, low = 0, high = array.length;
    while (low < high) {
      i = (low + high) >> 1;
      if (array[i] <= val)
        low = i+1;
      else
        high = i;
    }
    return low;
  }
  
  /* ======================================================================= */
  
  var diskData = null;
  
  var writeTrans = [
    0x96, 0x97, 0x9A, 0x9B, 0x9D, 0x9E, 0x9F, 0xA6,
    0xA7, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB2, 0xB3,
    0xB4, 0xB5, 0xB6, 0xB7, 0xB9, 0xBA, 0xBB, 0xBC, 
    0xBD, 0xBE, 0xBF, 0xCB, 0xCD, 0xCE, 0xCF, 0xD3,
    0xD6, 0xD7, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE,
    0xDF, 0xE5, 0xE6, 0xE7, 0xE9, 0xEA, 0xEB, 0xEC,
    0xED, 0xEE, 0xEF, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6,
    0xF7, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
  ];
  
  var readTrans = {};

  // Deskewing table for DOS 3.3 ordered disks  
  var physToLogical = [ 0, 7, 14, 6, 13, 5, 12, 4, 11, 3, 10, 2, 9, 1, 8, 15 ];
  
  // Overall state of the drive
  var motorOn = false;
  var isWarm = false;
  var coolTimer = null;
  var motorPhase = 0;
  
  // Track the head is currently sitting on
  var track = 0;
  var trackData = null;
  var trackTiming;
  var totalTrackTime;
  
  // Position and cycle within the track
  var timeBase;
  var pos;
  var cycle;
  
  // Write support
  var rwMode;
  var writeBaseTime;
  var curWriteByte;
  var nextWriteByte;
  var trackDirty;

  /* ======================================================================= */
  
  /** One-time initialization for the disk routines. */
  for (var i=0; i<64; i++)
    readTrans[writeTrans[i]] = i;
  
  /* ======================================================================= */
  
  this.init = function()
  {
    motorOn = false;
    isWarm = false;
    if (coolTimer != null)
        clearTimeout(coolTimer);
    coolTimer = null;
  }
  
  /** Take a DSK image and put it into the virtual drive */
  this.insert = function(data)
  {
    // Decode string data to bytes
    diskData = new Array(data.length);
    for (var i in data)
      diskData[i] = data.charCodeAt(i) & 0xFF;
    pos = cycle = 0;
    trackData = null;
    trackTiming = null;
    timeBase = t;
  }
  
  /** Remove a DSK image from the virtual drive, returning its data */
  this.eject = function() 
  {
    // Translate the data back to a string
    var ret = null;
    if (diskData != null) {
      var arr = new Array(diskData.length);
      for (var i in diskData)
        arr.push(String.fromCharCode(diskData[i]));
      ret = arr.join("");
    }
      
    diskData = null;
    trackData = null;
    trackTiming = null;
    
    return ret;
  }
  
  var outOff;
  
  /** Convert 256 8-bit bytes to 342 6-bit bytes, according to the 6/2 encoding */
  function nibblizeSector(inData, inOff)
  {
    // Write the nibble fragments
    var nibbles = new Array();
    var i;
    var val, v1, v2, v3;
    var prev = 0;
    for (i=0; i<86; i++) {
      v1 = inData[inOff+i];
      v2 = inData[inOff+i+86];
      v3 = inData[inOff+((i+172) & 0xFF)];
      val =  ((v1 & 2) >> 1) |
             ((v1 & 1) << 1) |
             ((v2 & 2) << 1) |
             ((v2 & 1) << 3) |
             ((v3 & 2) << 3) |
             ((v3 & 1) << 5);
      trackData[outOff++] = writeTrans[prev ^ val];
      prev = val;
    }
    
    // And translate the rest of the data
    for (i=0; i<256; i++) {
      val = inData[inOff+i] >> 2;
      trackData[outOff++] = writeTrans[prev ^ val];
      prev = val;
    }
    
    // Write the checksum byte last.
    trackData[outOff++] = writeTrans[prev];
  }
  
  /** Output some sync bytes. */
  function syncBytes(n) 
  {
    for (var i=0; i<n; i++)
      trackData[outOff++] = 0x3FF;
  }
  
  /** Output a byte as two 4/4 encoded bytes (no other translation necessary) */
  function encodeTwo(val)
  {
    trackData[outOff++] = ((val>>1) & 0x55) | 0xAA;
    trackData[outOff++] = (val & 0x55) | 0xAA;
  }
  
  /** Convert 16 sectors of data to raw nibbles. */
  function nibblizeTrack()
  {
    outOff = 0;
    
    trackData = new Array();
    trackDirty = false;
    
    // Start the track
    syncBytes(30); // at least 14, usually more than 40
    
    // Now process each physical sector
    for (var sector = 0; sector < 16; sector++)
    {
      // Emit the address field prologue
      trackData[outOff++] = 0xD5;
      trackData[outOff++] = 0xAA;
      trackData[outOff++] = 0x96;
      
      // Next the address data
      encodeTwo(254); // volume
      encodeTwo(track);
      encodeTwo(sector);
      encodeTwo(254 ^ track ^ sector); // checksum
      
      // And the address field epilogue
      trackData[outOff++] = 0xDE;
      trackData[outOff++] = 0xAA;
      trackData[outOff++] = 0xEB;
      
      // Sync bytes between address and data
      syncBytes(8); // between 5 and 10
      
      // Emit the data field prologue
      trackData[outOff++] = 0xD5;
      trackData[outOff++] = 0xAA;
      trackData[outOff++] = 0xAD;
      
      // Now comes the sector data itself
      var trackOffset = 16 * 256 * track;
      var logicalSector = physToLogical[sector];
      var secOffset = trackOffset + (256 * logicalSector);
      nibblizeSector(diskData, secOffset);
      
      // Followed by the data field epilogue
      trackData[outOff++] = 0xDE;
      trackData[outOff++] = 0xAA;
      trackData[outOff++] = 0xEB;

      // And some sync bytes between sectors
      if (sector < 15)
        syncBytes(20); // 14 to 24 bytes
    }
    
    //log("Nibblized track " + track);
  }
  
  /** Convert 342 6-bit bytes to 256 6-bit bytes, according to the 6/2 encoding */
  function denibblizeSector(inData, inOff)
  {
    // First read the nibble fragments
    var decoded = new Array();
    var i;
    var cur = 0;
    for (i=0; i<86; i++) {
      cur ^= readTrans[trackData[pos++]];
      decoded[i] = ((cur & 1) << 1) | ((cur & 2) >> 1);
      decoded[i+86] = ((cur & 4) >> 1) | ((cur & 8) >> 3);
      if (i+172 < 256)
        decoded[i+172] = ((cur & 16) >> 3) | ((cur & 32) >> 5);
    }
    
    // Then fill in the rest of the data
    for (i=0; i<256; i++) {
      cur ^= readTrans[trackData[pos++]];
      decoded[i] |= (cur << 2);
    }
    
    // Verify the checksum
    cur ^= readTrans[trackData[pos++]];
    if (cur != 0) {
      log("Warning: sector checksum mismatch.");
      return;
    }
  
    for (i=0; i<256; i++)
      inData[inOff+i] = decoded[i];
  }
  
  function denibblizeTrack()
  {
    // If existing data is not dirty, skip this.
    if (!trackDirty)
      return;
    
    var oldPos = pos;
    pos = 0;
    
    // Now process each physical sector
    for (var sector = 0; sector < 16; sector++)
    {
      // Find the data field prologue
      while (pos < trackData.length) {
        if (trackData[pos++] != 0xD5)
          continue;
        if (trackData[pos++] != 0xAA)
          continue;
        if (trackData[pos++] == 0xAD)
          break;
      }
      assert (pos < trackData.length); // corrupt?
      
      // Denibblize the sector data
      var trackOffset = 16 * 256 * track;
      var logicalSector = physToLogical[sector];
      var secOffset = trackOffset + (256 * logicalSector);
      denibblizeSector(diskData, secOffset);
    }
    
    // Data is clean now.
    log("Wrote track " + track);
    trackDirty = false;
    pos = oldPos;
  }

  /** Calculate cycle timing for all bytes on the track */
  function recalcTiming()
  {
    trackTiming = new Array();
    var tt = 0;
    for (var i = 0; i < trackData.length; i++) { // do use for-in here...
      trackTiming[i] = tt;
      tt += (trackData[i] > 255) ? 40 : 32;
    }
    trackTiming[i] = totalTrackTime = tt;        // ...because of this
  }
  
  var diskDebugCt = -1;
  
  /** Simulates a read from $C0EC, the read latch */
  function readByte()
  {
    if (diskData == null)
      return 0;
      
    // We need denibblized track data and timing data in all cases, but we
    // can lazily calculate it.
    if (trackData == null)
      nibblizeTrack();
    if (trackTiming === null)
      recalcTiming();
      
    // Decide what kind of skip we need
    var angle = t - timeBase;
    if ((angle >= totalTrackTime) || (angle - trackTiming[pos] > 200) ||
        (angle < trackTiming[pos]))
    {
      if (false && diskDebugCt > 0) {
        log("Long skip: angle=" + angle + ", totalTrackTime=" + totalTrackTime + 
            ", trackTiming[pos]=" + trackTiming[pos]);
      }
      
      // Long skip. Figure out number of rotations and adjust the base.
      timeBase += Math.floor(angle / totalTrackTime) * totalTrackTime;
      
      // Find new position in the track
      angle = t - timeBase;
      pos = binarySearch(trackTiming, angle) - 1;
    }
    else 
    {
      if (false && diskDebugCt > 0) {
        log("Short skip: angle=" + angle + ", totalTrackTime=" + totalTrackTime + 
            ", trackTiming[pos]=" + trackTiming[pos] +
            ", trackTiming[pos+1]=" + trackTiming[pos+1]);
      }
                    
      // Short skip forward in the current rotation
      while (angle >= trackTiming[pos+1])
        ++pos;
    }
    
//    if (pos == 0 && diskDebugCt <= 0) {
//      log("-----------------");
//      diskDebugCt = 500;
//    }
    
    // Now return the proper number of bits from the current byte
    var cycle = angle - trackTiming[pos];
    var nBits = trackData[pos] > 255 ? 10 : 8;
    
    // Keep the data valid for 8 cycles; apparently 4 isn't enough.
    var shift = nBits - (cycle >> 2) - 1;
    shift &= ~1;
    var ret = (trackData[pos] >> shift) & 0xFF;

    if (diskDebugCt > 0) {
      --diskDebugCt;
      log("disk read: pc=" + toHex(pc,4) + 
          ", angle=" + angle + ", pos=" + pos + 
          ", cycle=" + cycle + ", nBits=" + nBits + 
          ", data=" + toHex(trackData[pos], 2) +
          ", ret=" + toHex(ret, 2) + 
          (ret > 127 ? " ***" : ""));
    }
    
    assert(pos >= 0 && pos < trackData.length);
    assert(trackData[pos] !== undefined);
    assert(cycle >= 0 && cycle < 40);
    
    return ret;
  }
  
  function writeByte()
  {
    // Is there a pending write?
    if (curWriteByte >= 0) 
    {
      // How many cycles has it been?
      var elapsed = t - writeBaseTime;
      //log("Disk write: " + "old=" + toHex(trackData[pos], 2) + 
      //    ", new=" + toHex(curWriteByte, 2) +  ", elapsed=" + elapsed);
      if (elapsed >= 38 && elapsed < 42 && curWriteByte == 0xFF) 
      {
        // Self-sync byte. Make sure we line up with one on disk.
        while (trackData[pos] != 0x3FF)
          pos = (pos+1) % trackData.length;
      }
      else if (elapsed >= 30 && elapsed < 36)
      {
        // Data byte. Make sure to overwrite an existing data byte.
        while (trackData[pos] == 0x3FF)
          pos = (pos+1) % trackData.length;
        if (trackData[pos] != curWriteByte) {
          // ensure we don't overwrite addr fields
          assert(curWriteByte != 0xAA && curWriteByte != 0xD5); 
          trackDirty = true;
        }
        trackData[pos] = curWriteByte;
        pos = (pos+1) % trackData.length;
      }
      else
      {
        log("Invalid disk write: " + "old=" + toHex(trackData[pos], 2) + 
            ", new=" + toHex(curWriteByte, 2) +  ", elapsed=" + elapsed);
      }
    }
    else
      console.log("Hmm, wrote byte but didn't latch one?");
    
    // Queue this byte for writing after we know how many
    // cycles it sits in the latch.
    //
    curWriteByte = nextWriteByte;
    nextWriteByte = -1;
    writeBaseTime = t;    
  }
  
  /** 
   * Called back when the cool-down timer expires. Here is where
   * we actually shut off the motor and stop reading data.
   */
  function cooled() {
    //log("cooled");
    isWarm = false;
    motorOn = false;
    document.getElementById("diskLED").src = "images/red-off-32.png";
    clearTimeout(coolTimer);
    coolTimer = null;
  }
  
  function upTrack()
  {
    if (track == 34) {
      log("Disk head past end");
      return;
    }
    switchTrack(track+1);
    //log("Track up to " + track);
  }

  function downTrack()
  {
    if (track == 0) {
      //log("Disk recal");
      return;
    }
    switchTrack(track-1);
    //log("Track down to " + track);
  }

  function switchTrack(newTrack)
  {
    denibblizeTrack();
    track = newTrack;
    trackData = null; // will lazily calculate if needed
  }

  /** Read from one of the disk drive registers */
  this.softswitch_get = function(addr) 
  {
    switch (addr) {
      case 0xC0EC:
        if (isWarm)
          return rwMode == "read" ? readByte() : writeByte();
        else {
          if ((t - timeBase) > 300000) { // about 1/3 sec
            isWarm = true;
            //log("Warm t=" + t);
          }
        }
        break;
      case 0xC0E0:
        //log("Phase 0 off");
        break;
      case 0xC0E1:
        //log("Phase 0 on");
        if (motorPhase == 1)
          downTrack();
        else if (motorPhase == 3)
          upTrack();
        motorPhase = 0;
        break;
      case 0xC0E2:
        //log("Phase 1 off");
        break;
      case 0xC0E3:
        //log("Phase 1 on");
        motorPhase = 1;
        break;
      case 0xC0E4:
        //log("Phase 2 off");
        break;
      case 0xC0E5:
        //log("Phase 2 on");
        if (motorPhase == 1)
          upTrack();
        else if (motorPhase == 3)
          downTrack();
        motorPhase = 2;
        break;
      case 0xC0E6:
        //log("Phase 3 off");
        break;
      case 0xC0E7:
        //log("Phase 3 on");
        motorPhase = 3;
        break;
      case 0xC0E8:
        denibblizeTrack();
        if (coolTimer == null && motorOn)
          coolTimer = setTimeout(cooled, 1000); // 1 sec to allow further access
        break;
      case 0xC0E9:
        if (!motorOn) 
        {
          timeBase = t;
          pos = 0;
          //log("timeBase=" + t);
          // Start at random place on disk
          //timeBase = Math.max(0, t - Math.floor(Math.random() * totalTrackTime));
          motorOn = true;
          document.getElementById("diskLED").src = "images/red-on-32.png";
          rwMode = "read";
          curWriteByte = nextWriteByte = -1;
        }
        if (coolTimer != null) {
          clearTimeout(coolTimer); // if we were cooling, stop it -- get ready for more.
          coolTimer = null;
        }
        break;
      case 0xC0EE:
        rwMode = "read";
        break;
      case 0xC0EF:
        if (rwMode != "write") {
          rwMode = "write";
          curWriteByte = nextWriteByte = -1;
          writeBaseTime = t;
          readByte(); // set position on track
        }
        break;
    }
    return 0x77; // hopefully innocuous but telling
  };

  /** Write to one of the disk drive registers */
  this.softswitch_set = function(addr, val) 
  {
    this.softswitch_get(addr);
    if (addr == 0xC0ED)
      nextWriteByte = val;
  }
  
// end of scope
}
